<?php
/**
 * Email Certificate
 *
 * @package     AutomatorWP\Integrations\LearnDash\Actions\Email_Certificate
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LearnDash_Email_Certificate extends AutomatorWP_Integration_Action {

    public $integration = 'learndash';
    public $action = 'learndash_email_certificate';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var bool $result
     */
    public $result = false;

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Send a certificate through email', 'automatorwp-leardash' ),
            'select_option'     => __( 'Send <strong>a certificate</strong> through email', 'automatorwp-leardash' ),
            /* translators: %1$s: Cartificate. */
            'edit_label'        => sprintf( __( 'Send a %1$s through email', 'automatorwp-leardash' ), '{email}' ),
            /* translators: %1$s: Cartificate. */
            'log_label'         => sprintf( __( 'Send a %1$s through email', 'automatorwp-leardash' ), '{email}' ),
            'options'           => array(
                'email' => array(
                    'default' => __( 'certificate', 'automatorwp-leardash' ),
                    'fields' => array(
                        'post' => automatorwp_utilities_post_field( array(
                            'name'              => __( 'Certificate:', 'automatorwp-pro' ),
                            'option_none'       => false,
                            'option_custom'         => true,
                            'option_custom_desc'    => __( 'Certificate ID', 'automatorwp-pro' ),
                            'post_type'         => 'sfwd-certificates',
                            'placeholder'       => __( 'Select a certificate', 'automatorwp-pro' ),
                            'default'           => ''
                        ) ),
                        'post_custom' => automatorwp_utilities_custom_field( array(
                            'option_custom_desc'    => __( 'Certificate ID', 'automatorwp-pro' ),
                        ) ),
						'course' => automatorwp_utilities_post_field( array(
							'name'              => __( 'Course:', 'automatorwp-pro' ),
							'option_none_label' => __( 'all courses', 'automatorwp-pro' ),
							'option_custom'         => true,
							'option_custom_desc'    => __( 'Course ID', 'automatorwp-pro' ),
							'post_type'         => 'sfwd-courses',
						) ),
						'course_custom' => automatorwp_utilities_custom_field( array(
                            'option_custom_desc'    => __( 'Course ID', 'automatorwp-pro' ),
                        ) ),
                        'from' => array(
                            'name' => __( 'From:', 'automatorwp-leardash' ),
                            'desc' => __( 'Leave empty to use default WordPress email.', 'automatorwp-leardash' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'to' => array(
                            'name' => __( 'To:', 'automatorwp-leardash' ),
                            'desc' => __( 'Email address(es) to send the email. Accepts single or comma-separated list of emails.', 'automatorwp-leardash' )
                                . '<br>' . __( 'Leave empty to use the email of the user that completes the automation.', 'automatorwp-leardash' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                        'cc' => array(
                            'name' => __( 'CC:', 'automatorwp-leardash' ),
                            'desc' => __( 'Email address(es) that will receive a copy of this email. Accepts single or comma-separated list of emails.', 'automatorwp-leardash' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'bcc' => array(
                            'name' => __( 'BCC:', 'automatorwp-leardash' ),
                            'desc' => __( 'Email address(es) that will receive a copy of this email. Accepts single or comma-separated list of emails.', 'automatorwp-leardash' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'subject' => array(
                            'name' => __( 'Subject:', 'automatorwp-leardash' ),
                            'desc' => __( 'Email\'s subject.', 'automatorwp-leardash' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                        'content' => array(
                            'name' => __( 'Content:', 'automatorwp-leardash' ),
                            'desc' => __( 'Email\'s content.', 'automatorwp-leardash' ),
                            'type' => 'wysiwyg',
                            'required'  => true,
                            'default' => ''
                        ),
                    )
                )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $certificate_id     = $action_options['post'];
        $from               = $action_options['from'];
        $to                 = $action_options['to'];
        $cc                 = $action_options['cc'];
        $bcc                = $action_options['bcc'];
        $subject            = $action_options['subject'];
        $content            = $action_options['content'];
		$course_id = $action_options['course'];

        $certificate = get_post( $certificate_id );

        // Bail if can't find the certificate
        if( ! $certificate ) {
            return;
        }

        // Setup the vars used on LearnDash to store the PDF
        $_GET['destination'] = 'F';
        $_GET['destination_type'] = 'F';
        define( 'LEARNDASH_UPLOADS_BASE_DIR', WP_CONTENT_DIR . '/uploads/automatorwp-learndash/' );

        if ( ! file_exists( LEARNDASH_UPLOADS_BASE_DIR ) ) {
            mkdir( LEARNDASH_UPLOADS_BASE_DIR, 0755 );
        }

        // Include the post to PDF file
        require_once LEARNDASH_LMS_PLUGIN_DIR . '/includes/ld-convert-post-pdf.php';

        $filename = 'certificate-' . $certificate_id . '-' . time();

        // Create the PDF
        $certificate_args = array(
			'certificate_post' => $certificate_id,
            'save_path'        => LEARNDASH_UPLOADS_BASE_DIR, // Add save path.
			'file_name'        => $filename, // Add filename.
			'user'             => get_user_by( 'ID', $user_id ),
		);
        
        $file = automatorwp_learndash_generate_certificate( $certificate_args, '', $course_id, 'automatorwp' );

        //Bail if $file is false
        if ( (bool)$file === false ){
            return;
        }

        // Setup to
        if( empty( $to ) ) {
            $user = get_userdata( $user_id );
            $to = $user->user_email;
        }

        // Setup headers
        $headers = array();

        if( ! empty( $from ) ) {
            $headers[] = 'From: <' . $from . '>';
        }

        if ( ! empty( $cc ) ) {
            $headers[] = 'Cc: ' . $cc;
        }

        if ( ! empty( $bcc ) ) {
            $headers[] = 'Bcc: ' . $bcc;
        }

        $headers[] = 'Content-Type: text/html; charset='  . get_option( 'blog_charset' );

        // Setup the content
        $content = wpautop( $content );

        add_filter( 'wp_mail_content_type', 'automatorwp_set_html_content_type' );

        // Send email
        //$this->result = wp_mail( $to, $subject, $content, $headers, array( $file ) );
        $this->result = wp_mail( $to, $subject, $content, $headers, $file['message']  );

        remove_filter( 'wp_mail_content_type', 'automatorwp_set_html_content_type' );

        // Remove the certificate file
        unlink( $file['message'] );

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['from']       = $action_options['from'];
        $log_meta['to']         = $action_options['to'];
        $log_meta['cc']         = $action_options['cc'];
        $log_meta['bcc']        = $action_options['bcc'];
        $log_meta['subject']    = $action_options['subject'];
        $log_meta['content']    = $action_options['content'];
        $log_meta['result']     = ( $this->result ? __( 'Sent', 'automatorwp-leardash' ) : __( 'Not sent', 'automatorwp-leardash' ) );

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['email_info'] = array(
            'name' => __( 'Sending Information', 'automatorwp-leardash' ),
            'desc' => __( 'Information about email sent.', 'automatorwp-leardash' ),
            'type' => 'title',
        );

        $log_fields['from'] = array(
            'name' => __( 'From:', 'automatorwp-leardash' ),
            'desc' => __( 'Email sender.', 'automatorwp-leardash' ),
            'type' => 'text',
        );

        $log_fields['to'] = array(
            'name' => __( 'To:', 'automatorwp-leardash' ),
            'desc' => __( 'Email recipient.', 'automatorwp-leardash' ),
            'type' => 'text',
        );

        $log_fields['cc'] = array(
            'name' => __( 'CC:', 'automatorwp-leardash' ),
            'desc' => __( 'Carbon copy emails.', 'automatorwp-leardash' ),
            'type' => 'text',
        );

        $log_fields['bcc'] = array(
            'name' => __( 'BCC:', 'automatorwp-leardash' ),
            'desc' => __( 'Blind carbon copy emails.', 'automatorwp-leardash' ),
            'type' => 'text',
        );

        $log_fields['subject'] = array(
            'name' => __( 'Subject:', 'automatorwp-leardash' ),
            'desc' => __( 'Email\'s subject.', 'automatorwp-leardash' ),
            'type' => 'text',
        );

        $log_fields['content'] = array(
            'name' => __( 'Content:', 'automatorwp-leardash' ),
            'desc' => __( 'Email\'s content.', 'automatorwp-leardash' ),
            'type' => 'text',
            'wpautop' => true,
        );

        $log_fields['result'] = array(
            'name' => __( 'Sending result:', 'automatorwp-leardash' ),
            'desc' => __( 'If sending result is "Not sent" you need to check if your server\'s wp_mail() function is correctly configured.', 'automatorwp-leardash' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_LearnDash_Email_Certificate();